<?php
/**
 * Functions
 *
 * @package     AutomatorWP\Restrict_Content\Functions
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Options callback for select2 fields assigned to access levels
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_restrict_content_options_cb_access_level( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any access level', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }

        foreach( $value as $access_level_id ) {

            // Skip option none
            if( $access_level_id === $none_value ) {
                continue;
            }

            $options[$access_level_id] = automatorwp_restrict_content_get_access_level_title( $access_level_id );
        }
    }

    return $options;

}

/**
 * Get access levels
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_restrict_content_get_access_levels() {

    $access_levels = array();

    $all_levels = automatorwp_restrict_content_get_all_access_levels();
        
    foreach ( $all_levels as $access_level ){
    
        $access_levels[] = array(
            'id' => $access_level->id,
            'name' => $access_level->name,
        );
        
    }

    return $access_levels;
}

/**
 * Get the level title
 *
 * @since 1.0.0
 *
 * @param int $access_level_id
 *
 * @return string|null
 */
function automatorwp_restrict_content_get_access_level_title( $access_level_id ) {

    global $wpdb;

    // Empty title if no ID provided
    if( absint( $access_level_id ) === 0 ) {
        return '';
    }

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_restrict_content_access_levels' );

    // Search level by ID
    $access_level_name = $wpdb->get_var( $wpdb->prepare(
        "SELECT ul.name
        FROM {$ct_table->db->table_name} AS ul
        WHERE ul.id = %d",
        absint( $access_level_id ),
    ) );

    ct_reset_setup_table();

    return $access_level_name;

}

/**
 * Return an array of post types that Restrict Content is allowed to apply restrictions
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_restrict_content_post_types() {

    $post_types = array();

    $wp_post_types = get_post_types( array( 'public' => true ), 'objects' );

    // Excluded post types
    $excluded_post_types = array(
        'attachment',
    );

    /**
     * Filter post types to exclude to be restricted
     *
     * @since 1.0.0
     *
     * @pararm array $excluded_post_types By default, all public post types
     *
     * @return array
     */
    $excluded_post_types = apply_filters( 'automatorwp_restrict_content_excluded_post_types', $excluded_post_types );

    // Loop public post types
    foreach( $wp_post_types as $post_type ) {

        // Skip excluded post types
        if( in_array( $post_type->name, $excluded_post_types ) ) {
            continue;
        }

        // Setup an array like: 'post' => 'Posts'
        $post_types[$post_type->name] = $post_type->label;
    }

    /**
     * Filter post types that can be restricted
     *
     * @since 1.0.0
     *
     * @pararm array $post_types
     *
     * @return array
     */
    return apply_filters( 'automatorwp_restrict_content_post_types', $post_types );

}

/**
 * Return an array of post types slugs where AutomatorWP Restrict Content is allowed to apply restrictions
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_restrict_content_post_types_slugs() {

    $post_types = automatorwp_restrict_content_post_types();

    return array_keys( $post_types );
}

/**
 * Return an array of post types that Restrict Content is allowed to redirect (used for the redirect page option)
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_restrict_content_redirect_post_types() {

    $post_types = array();

    $wp_post_types = get_post_types( array( 'public' => true ), 'objects' );

    // Excluded post types
    $excluded_post_types = array(
        'attachment'
    );

    /**
     * Filter post types to exclude to be used for the redirect option
     *
     * @since 1.0.0
     *
     * @pararm array $excluded_post_types By default, all public post types except attachments
     *
     * @return array
     */
    $excluded_post_types = apply_filters( 'automatorwp_restrict_content_redirect_excluded_post_types', $excluded_post_types );

    // Loop public post types
    foreach( $wp_post_types as $post_type ) {

        // Skip excluded post types
        if( in_array( $post_type->name, $excluded_post_types ) ) {
            continue;
        }

        // Setup an array like: 'post' => 'Posts'
        $post_types[$post_type->name] = $post_type->label;
    }

    /**
     * Filter post types that can used for the redirect option
     *
     * @since 1.0.0
     *
     * @pararm array $post_types
     *
     * @return array
     */
    return apply_filters( 'automatorwp_restrict_content_redirect_post_types', $post_types );

}

/**
 * Return an array of post types slugs where AutomatorWP Restrict Content is allowed to apply restrictions
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_restrict_content_redirect_post_types_slugs() {

    $post_types = automatorwp_restrict_content_redirect_post_types();

    return array_keys( $post_types );
}

/**
 * Return an array of user roles as options to use on fields
 *
 * @since 1.0.2
 *
 * @return array
 */
function automatorwp_restrict_content_get_roles_options() {

    $options = array();

    $editable_roles = array_reverse( get_editable_roles() );

    foreach ( $editable_roles as $role => $details ) {

        // Skip administrator roles
        if( $role === 'administrator' ) {
            continue;
        }

        $options[$role] = translate_user_role( $details['name'] );

    }

    return $options;
}

/**
 * Return an array of user roles as options to use on fields
 *
 * @since 1.0.2
 *
 * @return array
 */
function automatorwp_restrict_content_get_access_levels_options() {

    $options = array();

    $access_levels = automatorwp_restrict_content_get_all_access_levels();

    foreach ( $access_levels as $access_level ) {

        $options[$access_level->id] = $access_level->name;

    }

    return $options;
}